#!/usr/bin/env node
// Database migration script for MySQL

const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');
require('dotenv').config({ path: '.env.local' });

// Database configuration
const config = {
  host: process.env.DB_HOST || 'localhost',
  user: process.env.DB_USER || 'root',
  password: process.env.DB_PASSWORD || '',
  database: process.env.DB_NAME || 'textile_manager',
  port: parseInt(process.env.DB_PORT || '3306'),
  multipleStatements: true,
};

async function runMigration() {
  let connection;
  
  try {
    console.log('🔄 Connecting to MySQL database...');
    connection = await mysql.createConnection(config);
    console.log('✅ Connected to database successfully');

    // Read and execute schema file
    const schemaPath = path.join(__dirname, '../database/schema.sql');
    if (fs.existsSync(schemaPath)) {
      console.log('📄 Reading schema.sql...');
      const schema = fs.readFileSync(schemaPath, 'utf8');
      
      console.log('🔄 Executing schema...');
      await connection.execute(schema);
      console.log('✅ Schema executed successfully');
    }

    // Read and execute migration files
    const migrationsDir = path.join(__dirname, '../database/migrations');
    if (fs.existsSync(migrationsDir)) {
      const migrationFiles = fs.readdirSync(migrationsDir)
        .filter(file => file.endsWith('.sql'))
        .sort();

      for (const file of migrationFiles) {
        console.log(`🔄 Running migration: ${file}`);
        const migrationPath = path.join(migrationsDir, file);
        const migration = fs.readFileSync(migrationPath, 'utf8');
        
        await connection.execute(migration);
        console.log(`✅ Migration ${file} completed`);
      }
    }

    console.log('🎉 All migrations completed successfully!');

  } catch (error) {
    console.error('❌ Migration failed:', error.message);
    
    if (error.code === 'ECONNREFUSED') {
      console.error('💡 Make sure MySQL is running and connection details are correct');
    } else if (error.code === 'ER_ACCESS_DENIED_ERROR') {
      console.error('💡 Check your database username and password');
    } else if (error.code === 'ER_BAD_DB_ERROR') {
      console.error('💡 Database does not exist. Create it first or check the database name');
    }
    
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
      console.log('🔌 Database connection closed');
    }
  }
}

// Check if database exists, create if not
async function createDatabaseIfNotExists() {
  const configWithoutDB = { ...config };
  delete configWithoutDB.database;
  
  let connection;
  
  try {
    connection = await mysql.createConnection(configWithoutDB);
    
    // Create database if it doesn't exist
    await connection.execute(`CREATE DATABASE IF NOT EXISTS \`${config.database}\` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci`);
    console.log(`✅ Database '${config.database}' is ready`);
    
  } catch (error) {
    console.error('❌ Failed to create database:', error.message);
    throw error;
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

// Main execution
async function main() {
  console.log('🚀 Starting database migration...');
  console.log(`📍 Target: ${config.host}:${config.port}/${config.database}`);
  
  try {
    await createDatabaseIfNotExists();
    await runMigration();
  } catch (error) {
    console.error('💥 Migration process failed');
    process.exit(1);
  }
}

// Run if called directly
if (require.main === module) {
  main();
}

module.exports = { runMigration, createDatabaseIfNotExists };
