#!/usr/bin/env node
// Database seeder script for MySQL

const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');
require('dotenv').config({ path: '.env.local' });

// Database configuration
const config = {
  host: process.env.DB_HOST || 'localhost',
  user: process.env.DB_USER || 'root',
  password: process.env.DB_PASSWORD || '',
  database: process.env.DB_NAME || 'textile_manager',
  port: parseInt(process.env.DB_PORT || '3306'),
  multipleStatements: true,
};

async function seedDatabase() {
  let connection;
  
  try {
    console.log('🔄 Connecting to MySQL database...');
    connection = await mysql.createConnection(config);
    console.log('✅ Connected to database successfully');

    // Read and execute seeder files
    const seedersDir = path.join(__dirname, '../database/seeders');
    if (fs.existsSync(seedersDir)) {
      const seederFiles = fs.readdirSync(seedersDir)
        .filter(file => file.endsWith('.sql'))
        .sort();

      for (const file of seederFiles) {
        console.log(`🌱 Running seeder: ${file}`);
        const seederPath = path.join(seedersDir, file);
        const seeder = fs.readFileSync(seederPath, 'utf8');
        
        await connection.execute(seeder);
        console.log(`✅ Seeder ${file} completed`);
      }
    }

    // Create default admin user
    await createDefaultAdmin(connection);

    console.log('🎉 Database seeding completed successfully!');

  } catch (error) {
    console.error('❌ Seeding failed:', error.message);
    
    if (error.code === 'ECONNREFUSED') {
      console.error('💡 Make sure MySQL is running and connection details are correct');
    } else if (error.code === 'ER_ACCESS_DENIED_ERROR') {
      console.error('💡 Check your database username and password');
    } else if (error.code === 'ER_BAD_DB_ERROR') {
      console.error('💡 Database does not exist. Run migration first');
    }
    
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
      console.log('🔌 Database connection closed');
    }
  }
}

async function createDefaultAdmin(connection) {
  try {
    console.log('👤 Creating default admin user...');
    
    // Check if admin user already exists
    const [existingUsers] = await connection.execute(
      'SELECT id FROM users WHERE email = ? LIMIT 1',
      ['admin@textile.com']
    );

    if (existingUsers.length > 0) {
      console.log('ℹ️  Admin user already exists, skipping...');
      return;
    }

    // Create default shop first
    const [shopResult] = await connection.execute(`
      INSERT INTO shops (shop_name, email, address, contact) 
      VALUES (?, ?, ?, ?)
    `, [
      'Demo Tailor Shop',
      'contact@textile.com',
      '123 Main Street, City, State',
      '+1234567890'
    ]);

    const shopId = shopResult.insertId;

    // Hash password (simple hash for demo - use bcrypt in production)
    const bcrypt = require('bcrypt');
    const hashedPassword = await bcrypt.hash('admin123', 10);

    // Create admin user
    const [userResult] = await connection.execute(`
      INSERT INTO users (email, password_hash, display_name, role, shop_id, is_active) 
      VALUES (?, ?, ?, ?, ?, ?)
    `, [
      'admin@textile.com',
      hashedPassword,
      'Admin User',
      'owner',
      shopId,
      true
    ]);

    const userId = userResult.insertId;

    // Update shop with owner_id
    await connection.execute(
      'UPDATE shops SET owner_id = ? WHERE id = ?',
      [userId, shopId]
    );

    console.log('✅ Default admin user created:');
    console.log('   Email: admin@textile.com');
    console.log('   Password: admin123');
    console.log('   Shop ID:', shopId);
    console.log('⚠️  Please change the password after first login!');

  } catch (error) {
    console.error('❌ Failed to create admin user:', error.message);
    throw error;
  }
}

// Check database status
async function checkDatabaseStatus() {
  let connection;
  
  try {
    connection = await mysql.createConnection(config);
    
    // Check if tables exist
    const [tables] = await connection.execute(`
      SELECT TABLE_NAME 
      FROM INFORMATION_SCHEMA.TABLES 
      WHERE TABLE_SCHEMA = ? 
      AND TABLE_NAME IN ('users', 'shops', 'customers', 'orders')
    `, [config.database]);

    if (tables.length < 4) {
      console.log('⚠️  Database tables not found. Please run migration first:');
      console.log('   npm run db:migrate');
      process.exit(1);
    }

    console.log('✅ Database structure verified');

  } catch (error) {
    console.error('❌ Database check failed:', error.message);
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

// Main execution
async function main() {
  console.log('🌱 Starting database seeding...');
  console.log(`📍 Target: ${config.host}:${config.port}/${config.database}`);
  
  try {
    await checkDatabaseStatus();
    await seedDatabase();
  } catch (error) {
    console.error('💥 Seeding process failed');
    process.exit(1);
  }
}

// Run if called directly
if (require.main === module) {
  main();
}

module.exports = { seedDatabase, createDefaultAdmin };
