'use client';

import React, { useState, useEffect, useMemo } from 'react';
import { useRouter } from 'next/navigation';
import { MoreHorizontal, Search, ChevronDown, Trash2, File, Upload } from "lucide-react";
import { format, startOfMonth, endOfMonth, eachMonthOfInterval, subMonths } from 'date-fns';

import { Button } from "@/components/ui/button";
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Input } from '@/components/ui/input';
import { PageHeader } from "@/components/page-header";
import { getMockEmployees, getMockOrders, getOtherExpenses, type Employee, type Order, type Expense } from "@/lib/data";
import { EmployeeDialog } from '@/components/employees/employee-dialog';
import { AddExpenseDialog } from '@/components/expenses/add-expense-dialog';
import { ExpenseFormValues } from '@/components/expenses/add-expense-form';
import { Checkbox } from '@/components/ui/checkbox';
import { exportToCsv } from '@/lib/utils';
import { ImportEmployeesDialog } from '@/components/employees/import-employees-dialog';


export default function EmployeesPage() {
    const [employees, setEmployees] = useState<Employee[]>([]);
    const [orders, setOrders] = useState<Order[]>([]);
    const [otherExpenses, setOtherExpenses] = useState<Expense[]>([]);
    const [searchTerm, setSearchTerm] = useState('');
    const [isEmployeeDialogOpen, setIsEmployeeDialogOpen] = useState(false);
    const [isAdvanceDialogOpen, setIsAdvanceDialogOpen] = useState(false);
    const [isImportDialogOpen, setIsImportDialogOpen] = useState(false);
    const [editingEmployee, setEditingEmployee] = useState<Employee | null>(null);
    const [advancingEmployee, setAdvancingEmployee] = useState<Employee | null>(null);
    const [deletingEmployee, setDeletingEmployee] = useState<Employee | null>(null);
    const [selectedItems, setSelectedItems] = useState<Set<string>>(new Set());
    const [selectedMonth, setSelectedMonth] = useState<Date>(startOfMonth(new Date()));
    const router = useRouter();


    useEffect(() => {
        loadData();
         const handleStorageChange = () => {
            loadData();
        };

        window.addEventListener('storage', handleStorageChange);
        return () => {
            window.removeEventListener('storage', handleStorageChange);
        };
    }, []);
    
    const loadData = () => {
        setEmployees(getMockEmployees());
        setOrders(getMockOrders());
        setOtherExpenses(getOtherExpenses());
    }

    const updateEmployees = (newEmployees: Employee[]) => {
        localStorage.setItem('mockEmployees', JSON.stringify(newEmployees));
        loadData();
        setSelectedItems(new Set());
    };
    
    const updateOtherExpenses = (newExpenses: Expense[]) => {
        localStorage.setItem('mockOtherExpenses', JSON.stringify(newExpenses));
        loadData();
    };

    const handleEmployeeFormSubmit = (data: Omit<Employee, 'id'>) => {
        if (editingEmployee) {
            const updatedEmployees = employees.map(emp => emp.id === editingEmployee.id ? { ...emp, ...data } : emp);
            updateEmployees(updatedEmployees);
        } else {
            const newEmployee: Employee = {
                id: `EMP${(employees.length + 1).toString().padStart(3, '0')}`,
                ...data,
            };
            updateEmployees([newEmployee, ...employees]);
        }
        closeDialogs();
    };
    
    const handleAdvanceFormSubmit = (data: ExpenseFormValues) => {
        const currentExpenses = getOtherExpenses();
        const newExpense: Expense = {
            id: `exp-adv-${Date.now()}`,
            ...data,
            isEditable: true,
        };
        updateOtherExpenses([newExpense, ...currentExpenses]);
        closeDialogs();
    };


    const handleDelete = (employee: Employee) => {
        const newEmployees = employees.filter(emp => emp.id !== employee.id);
        updateEmployees(newEmployees);
        setDeletingEmployee(null);
    };

    const handleBulkDelete = () => {
        const newEmployees = employees.filter(emp => !selectedItems.has(emp.id));
        updateEmployees(newEmployees);
        setDeletingEmployee(null);
    };
    
    const openAddDialog = () => {
        setEditingEmployee(null);
        setIsEmployeeDialogOpen(true);
    };

    const openEditDialog = (employee: Employee) => {
        setEditingEmployee(employee);
        setIsEmployeeDialogOpen(true);
    };

    const openAdvanceDialog = (employee: Employee) => {
        setAdvancingEmployee(employee);
        setIsAdvanceDialogOpen(true);
    };

    const openDeleteDialog = (employee: Employee) => {
        setDeletingEmployee(employee);
    };

    const openBulkDeleteDialog = () => {
        setDeletingEmployee({ id: 'bulk-delete', name: `${selectedItems.size} employees` } as any);
    };
    
    const viewProfile = (employeeId: string) => {
        router.push(`/dashboard/employees/${employeeId}`);
    };

    const closeDialogs = () => {
        setIsEmployeeDialogOpen(false);
        setIsAdvanceDialogOpen(false);
        setEditingEmployee(null);
        setAdvancingEmployee(null);
    };
    
    const handleExport = () => {
        exportToCsv(filteredEmployees.map(e => ({id: e.id, name: e.name, phone: e.phone, role: e.role, monthlySalary: e.monthlySalary || 0})), 'employees.csv');
    };
    
    const handleImport = (newItems: Employee[]) => {
        const updatedEmployees = [...employees];
        newItems.forEach(newItem => {
            const existingIndex = updatedEmployees.findIndex(i => i.id === newItem.id);
            if (existingIndex > -1) {
                updatedEmployees[existingIndex] = newItem;
            } else {
                updatedEmployees.push(newItem);
            }
        });
        updateEmployees(updatedEmployees);
        setIsImportDialogOpen(false);
    };
    
    const handleSelectAll = (checked: boolean) => {
        if (checked) {
        setSelectedItems(new Set(filteredEmployees.map(item => item.id)));
        } else {
        setSelectedItems(new Set());
        }
    };

    const handleSelectItem = (itemId: string, checked: boolean) => {
        const newSelection = new Set(selectedItems);
        if (checked) {
        newSelection.add(itemId);
        } else {
        newSelection.delete(itemId);
        }
        setSelectedItems(newSelection);
    };


    const monthOptions = eachMonthOfInterval({
        start: subMonths(new Date(), 11),
        end: new Date(),
    }).reverse();

    const employeeStats = useMemo(() => {
        const monthStart = startOfMonth(selectedMonth);
        const monthEnd = endOfMonth(selectedMonth);

        return employees.map(employee => {
            const monthlyAdvances = otherExpenses
                .filter(exp => 
                    exp.category === 'Salary Advance' &&
                    exp.person === employee.name &&
                    new Date(exp.date) >= monthStart &&
                    new Date(exp.date) <= monthEnd
                )
                .reduce((sum, exp) => sum + exp.amount, 0);

            if (employee.role === 'Tailor') {
                 const totalEarned = orders
                    .filter(o => {
                        const completionDate = o.artisanCompletedDate ? new Date(o.artisanCompletedDate) : null;
                        return (
                            o.artisan === employee.name &&
                            completionDate &&
                            completionDate >= monthStart &&
                            completionDate <= monthEnd
                        );
                    })
                    .reduce((sum, order) => sum + order.totalArtisanWage, 0);

                const ordersCompleted = orders.filter(o => o.artisan === employee.name && new Date(o.date) >= monthStart && new Date(o.date) <= monthEnd).length;
                const balance = totalEarned - monthlyAdvances;

                return { ...employee, ordersCompleted, totalEarned, advancePaid: monthlyAdvances, balance };
            } else {
                 const balance = (employee.monthlySalary || 0) - monthlyAdvances;
                 return { ...employee, balance, advancePaid: monthlyAdvances, ordersCompleted: 0, totalEarned: employee.monthlySalary || 0 };
            }
        });
    }, [employees, orders, otherExpenses, selectedMonth]);


    const filteredEmployees = employeeStats.filter(employee =>
        employee.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        employee.phone.includes(searchTerm)
    );

  return (
    <div className="flex flex-col gap-4">
        <PageHeader title="Employees" actionButtonText="Add Employee" onActionButtonClick={openAddDialog}>
            <div className="relative w-full max-w-sm">
                <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
                <Input
                    type="search"
                    placeholder="Search employees..."
                    className="pl-8 w-full"
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                />
            </div>
             <DropdownMenu>
                <DropdownMenuTrigger asChild>
                <Button variant="outline" size="sm" className="h-8 gap-1">
                    <span>{selectedMonth ? format(selectedMonth, 'MMMM yyyy') : 'Filter by month'}</span>
                    <ChevronDown className="h-3.5 w-3.5" />
                </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                {monthOptions.map((month) => (
                    <DropdownMenuItem key={month.toString()} onSelect={() => setSelectedMonth(month)}>
                    {format(month, 'MMMM yyyy')}
                    </DropdownMenuItem>
                ))}
                </DropdownMenuContent>
            </DropdownMenu>
             {selectedItems.size > 0 && (
                <Button size="sm" variant="destructive" className="h-8 gap-1" onClick={openBulkDeleteDialog}>
                    <Trash2 className="h-3.5 w-3.5" />
                    <span>Delete ({selectedItems.size})</span>
                </Button>
            )}
             <Button size="sm" variant="outline" className="h-8 gap-1" onClick={() => setIsImportDialogOpen(true)}>
                <Upload className="h-3.5 w-3.5" />
                <span className="sr-only sm:not-sr-only sm:whitespace-nowrap">
                Import
                </span>
            </Button>
            <Button size="sm" variant="outline" className="h-8 gap-1" onClick={handleExport}>
                <File className="h-3.5 w-3.5" />
                <span className="sr-only sm:not-sr-only sm:whitespace-nowrap">
                Export
                </span>
            </Button>
        </PageHeader>
        
        <EmployeeDialog
            isOpen={isEmployeeDialogOpen}
            onOpenChange={setIsEmployeeDialogOpen}
            employee={editingEmployee}
            onSubmit={handleEmployeeFormSubmit}
        />
        
        {advancingEmployee && (
            <AddExpenseDialog
                isOpen={isAdvanceDialogOpen}
                onOpenChange={setIsAdvanceDialogOpen}
                expense={null}
                onSubmit={handleAdvanceFormSubmit}
                employees={employees}
                defaultCategory="Salary Advance"
                defaultPerson={advancingEmployee.name}
            />
        )}
        
        <ImportEmployeesDialog
            isOpen={isImportDialogOpen}
            onOpenChange={setIsImportDialogOpen}
            onImport={handleImport}
        />


        <AlertDialog open={!!deletingEmployee} onOpenChange={() => setDeletingEmployee(null)}>
            <AlertDialogContent>
                <AlertDialogHeader>
                    <AlertDialogTitle>Are you sure?</AlertDialogTitle>
                    <AlertDialogDescription>
                        This will permanently delete {deletingEmployee?.id === 'bulk-delete' ? `the selected ${deletingEmployee.name}` : `the employee: ${deletingEmployee?.name}`}. This action cannot be undone.
                    </AlertDialogDescription>
                </AlertDialogHeader>
                <AlertDialogFooter>
                    <AlertDialogCancel>Cancel</AlertDialogCancel>
                    <AlertDialogAction onClick={() => {
                        if (deletingEmployee?.id === 'bulk-delete') {
                            handleBulkDelete();
                        } else if (deletingEmployee) {
                            handleDelete(deletingEmployee);
                        }
                    }}>Delete</AlertDialogAction>
                </AlertDialogFooter>
            </AlertDialogContent>
       </AlertDialog>

        <Card>
            <CardHeader>
                <CardTitle>Employee Management</CardTitle>
                <CardDescription>
                View, manage, and track your staff salaries and payments for {format(selectedMonth, 'MMMM yyyy')}.
                </CardDescription>
            </CardHeader>
            <CardContent>
                <Table>
                <TableHeader>
                    <TableRow>
                    <TableHead className="w-[40px] pl-4">
                        <Checkbox
                            checked={selectedItems.size > 0 && selectedItems.size === filteredEmployees.length}
                            onCheckedChange={(checked) => handleSelectAll(!!checked)}
                            aria-label="Select all"
                        />
                    </TableHead>
                    <TableHead>Name</TableHead>
                    <TableHead>Role</TableHead>
                    <TableHead className="hidden md:table-cell">Phone</TableHead>
                    <TableHead className="hidden sm:table-cell">Salary/Earnings</TableHead>
                    <TableHead className="hidden sm:table-cell">Advance Paid</TableHead>
                    <TableHead className="text-right">Balance</TableHead>
                    <TableHead>
                        <span className="sr-only">Actions</span>
                    </TableHead>
                    </TableRow>
                </TableHeader>
                <TableBody>
                    {filteredEmployees.map((employee) => {
                        const isSelected = selectedItems.has(employee.id);
                        return (
                            <TableRow key={employee.id} data-state={isSelected ? 'selected' : ''}>
                                <TableCell className="pl-4">
                                    <Checkbox
                                        checked={isSelected}
                                        onCheckedChange={(checked) => handleSelectItem(employee.id, !!checked)}
                                        aria-label={`Select item ${employee.name}`}
                                    />
                                </TableCell>
                                <TableCell>
                                    <div className="font-medium">{employee.name}</div>
                                    <div className="text-sm text-muted-foreground sm:hidden">৳{(employee.balance || 0).toFixed(2)}</div>
                                </TableCell>
                                <TableCell>{employee.role}</TableCell>
                                <TableCell className="hidden md:table-cell">{employee.phone}</TableCell>
                                
                                {employee.role === 'Tailor' ? (
                                    <TableCell className="hidden sm:table-cell">
                                        <div className="font-medium">৳{(employee.totalEarned || 0).toFixed(2)}</div>
                                        <div className="text-xs text-muted-foreground">{employee.ordersCompleted} orders</div>
                                    </TableCell>
                                ) : (
                                    <TableCell className="hidden sm:table-cell">৳{(employee.monthlySalary || 0).toFixed(2)}</TableCell>
                                )}
                                
                                <TableCell className="hidden sm:table-cell">৳{(employee.advancePaid || 0).toFixed(2)}</TableCell>
                                <TableCell className="text-right">৳{(employee.balance || 0).toFixed(2)}</TableCell>
                                
                                <TableCell>
                                <DropdownMenu>
                                    <DropdownMenuTrigger asChild>
                                    <Button
                                        aria-haspopup="true"
                                        size="icon"
                                        variant="ghost"
                                    >
                                        <MoreHorizontal className="h-4 w-4" />
                                        <span className="sr-only">Toggle menu</span>
                                    </Button>
                                    </DropdownMenuTrigger>
                                    <DropdownMenuContent align="end">
                                        <DropdownMenuLabel>Actions</DropdownMenuLabel>
                                        <DropdownMenuItem onClick={() => openEditDialog(employee)}>Edit</DropdownMenuItem>
                                        <DropdownMenuItem onClick={() => viewProfile(employee.id)}>View Profile</DropdownMenuItem>
                                        <DropdownMenuItem onClick={() => openAdvanceDialog(employee)}>Add Advance</DropdownMenuItem>
                                        <DropdownMenuItem className="text-red-600" onClick={() => openDeleteDialog(employee)}>Delete</DropdownMenuItem>
                                    </DropdownMenuContent>
                                </DropdownMenu>
                                </TableCell>
                            </TableRow>
                        )
                    })}
                </TableBody>
                </Table>
            </CardContent>
             <CardFooter>
                <div className="text-xs text-muted-foreground">
                Showing <strong>1-{filteredEmployees.length}</strong> of <strong>{employees.length}</strong> employees
                </div>
            </CardFooter>
        </Card>
    </div>
  )
}
