'use client';

import { useState, useEffect } from 'react';
import { File, ChevronDown, MoreHorizontal } from "lucide-react"
import { Button } from "@/components/ui/button"
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card"
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog"

import { getMockIncome, getMockOrders, type Transaction } from "@/lib/data"
import { PageHeader } from "@/components/page-header"
import { AddIncomeDialog } from '@/components/income/add-income-dialog';
import { exportToCsv } from '@/lib/utils';
import { format, subMonths, startOfMonth, endOfMonth, eachMonthOfInterval } from 'date-fns';
import { TransactionFormValues } from '@/components/income/add-income-form';

export default function IncomePage() {
  const [allIncomeTransactions, setAllIncomeTransactions] = useState<Transaction[]>([]);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingIncome, setEditingIncome] = useState<Transaction | null>(null);
  const [deletingIncome, setDeletingIncome] = useState<Transaction | null>(null);
  const [selectedMonth, setSelectedMonth] = useState<Date | null>(null);

  const loadIncomeData = () => {
    const orders = getMockOrders();
    const otherIncome = getMockIncome();

    const incomeFromOrders: Transaction[] = orders
      .filter(order => order.paidAmount > 0)
      .map(order => ({
        id: `inc-ord-${order.id}`,
        category: 'Order Payment',
        date: order.date,
        amount: order.paidAmount,
        description: `Payment for Order #${order.id} from ${order.customer}`,
        isEditable: false,
      }));
    
    const editableIncome = otherIncome.map(i => ({ ...i, isEditable: true }));
    const allIncome = [...incomeFromOrders, ...editableIncome].sort((a, b) => new Date(b.date).getTime() - new Date(a.date).getTime());
    
    setAllIncomeTransactions(allIncome);
  }

  useEffect(() => {
    loadIncomeData();

    const handleStorageChange = () => {
      loadIncomeData();
    };
    
    window.addEventListener('storage', handleStorageChange);
    return () => {
      window.removeEventListener('storage', handleStorageChange);
    };
  }, []);

  const updateOtherIncome = (newIncome: Transaction[]) => {
    localStorage.setItem('mockIncome', JSON.stringify(newIncome.filter(i => i.isEditable)));
    loadIncomeData();
  };


  const handleFormSubmit = (data: TransactionFormValues) => {
    const otherIncome = getMockIncome();
    if (editingIncome) {
        const updatedIncome = otherIncome.map(inc => inc.id === editingIncome.id ? { ...inc, ...data } : inc);
        updateOtherIncome(updatedIncome);
    } else {
        const newIncome: Transaction = {
            id: `inc-misc-${Date.now()}`,
            ...data,
        };
        updateOtherIncome([newIncome, ...otherIncome]);
    }
    closeDialogs();
  };

  const handleDelete = () => {
    if (deletingIncome) {
        const otherIncome = getMockIncome();
        const newIncome = otherIncome.filter(inc => inc.id !== deletingIncome.id);
        updateOtherIncome(newIncome);
        setDeletingIncome(null);
    }
  };

  const openAddDialog = () => {
      setEditingIncome(null);
      setIsDialogOpen(true);
  };

  const openEditDialog = (income: Transaction) => {
      setEditingIncome(income);
      setIsDialogOpen(true);
  };

  const openDeleteDialog = (income: Transaction) => {
      setDeletingIncome(income);
  };

  const closeDialogs = () => {
      setIsDialogOpen(false);
      setEditingIncome(null);
  };


  const handleExport = () => {
    exportToCsv(filteredIncomeTransactions, 'income.csv');
  };

  const monthOptions = eachMonthOfInterval({
    start: subMonths(new Date(), 11),
    end: new Date(),
  }).reverse();

  const filteredIncomeTransactions = selectedMonth 
    ? allIncomeTransactions.filter(transaction => {
        const transactionDate = new Date(transaction.date);
        return transactionDate >= startOfMonth(selectedMonth) && transactionDate <= endOfMonth(selectedMonth);
    })
    : allIncomeTransactions;

  return (
    <div className="flex flex-col gap-4">
        <PageHeader title="Income" actionButtonText="Add Income" onActionButtonClick={openAddDialog}>
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="outline" className="h-8 gap-1">
                  <span>{selectedMonth ? format(selectedMonth, 'MMMM yyyy') : 'Filter by month'}</span>
                  <ChevronDown className="h-3.5 w-3.5" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end">
                <DropdownMenuItem onSelect={() => setSelectedMonth(null)}>All Months</DropdownMenuItem>
                {monthOptions.map((month) => (
                  <DropdownMenuItem key={month.toString()} onSelect={() => setSelectedMonth(month)}>
                    {format(month, 'MMMM yyyy')}
                  </DropdownMenuItem>
                ))}
              </DropdownMenuContent>
            </DropdownMenu>
          <Button size="sm" variant="outline" className="h-8 gap-1" onClick={handleExport}>
            <File className="h-3.5 w-3.5" />
            <span className="sr-only sm:not-sr-only sm:whitespace-nowrap">
              Export
            </span>
          </Button>
      </PageHeader>

       <AddIncomeDialog
        isOpen={isDialogOpen}
        onOpenChange={setIsDialogOpen}
        income={editingIncome}
        onSubmit={handleFormSubmit}
      />
      
       <AlertDialog open={!!deletingIncome} onOpenChange={() => setDeletingIncome(null)}>
            <AlertDialogContent>
                <AlertDialogHeader>
                    <AlertDialogTitle>Are you sure?</AlertDialogTitle>
                    <AlertDialogDescription>
                        This will permanently delete this income record. This action cannot be undone.
                    </AlertDialogDescription>
                </AlertDialogHeader>
                <AlertDialogFooter>
                    <AlertDialogCancel>Cancel</AlertDialogCancel>
                    <AlertDialogAction onClick={handleDelete}>Delete</AlertDialogAction>
                </AlertDialogFooter>
            </AlertDialogContent>
       </AlertDialog>


      <Card>
        <CardHeader>
          <CardTitle>Income Transactions</CardTitle>
          <CardDescription>
            A log of all income received by your business, including order payments.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Category</TableHead>
                <TableHead className="hidden sm:table-cell">Description</TableHead>
                <TableHead className="hidden md:table-cell">Date</TableHead>
                <TableHead className="text-right">Amount</TableHead>
                <TableHead>
                    <span className="sr-only">Actions</span>
                </TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredIncomeTransactions.map((income) => (
                <TableRow key={income.id}>
                  <TableCell className="font-medium">{income.category}</TableCell>
                  <TableCell className="hidden sm:table-cell">{income.description}</TableCell>
                  <TableCell className="hidden md:table-cell">{new Date(income.date).toLocaleDateString()}</TableCell>
                  <TableCell className="text-right text-green-600">+৳{income.amount.toFixed(2)}</TableCell>
                  <TableCell>
                    {income.isEditable && (
                        <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                            <Button
                                aria-haspopup="true"
                                size="icon"
                                variant="ghost"
                            >
                                <MoreHorizontal className="h-4 w-4" />
                                <span className="sr-only">Toggle menu</span>
                            </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                                <DropdownMenuLabel>Actions</DropdownMenuLabel>
                                <DropdownMenuItem onClick={() => openEditDialog(income)}>Edit</DropdownMenuItem>
                                <DropdownMenuItem className="text-red-600" onClick={() => openDeleteDialog(income)}>Delete</DropdownMenuItem>
                            </DropdownMenuContent>
                        </DropdownMenu>
                    )}
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </CardContent>
        <CardFooter>
          <div className="text-xs text-muted-foreground">
            Showing <strong>1-{filteredIncomeTransactions.length}</strong> of <strong>{allIncomeTransactions.length}</strong> transactions
          </div>
        </CardFooter>
      </Card>
    </div>
  )
}
