'use client';

import React, { useState, useEffect } from 'react';
import { File, ListFilter, MoreHorizontal, Printer, Search, ChevronDown, Trash2, Upload } from "lucide-react"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card"
import {
  DropdownMenu,
  DropdownMenuCheckboxItem,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
  DropdownMenuSub,
  DropdownMenuSubTrigger,
  DropdownMenuSubContent
} from "@/components/ui/dropdown-menu"
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table"
import {
  Tabs,
  TabsContent,
} from "@/components/ui/tabs"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog"

import { getMockOrders, type Order, getMockCustomers, type Customer, getArtisans, getCuttingMasters, Employee, getMockTailoringItems, TailoringItem, getShopProfile, ShopProfile, getMockMeasurements, addMockMeasurement, updateMockMeasurement } from "@/lib/data"
import { PageHeader } from "@/components/page-header"
import { OrderDialog } from '@/components/orders/order-dialog';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';
import { Input } from '@/components/ui/input';
import { format, subMonths, startOfMonth, endOfMonth, eachMonthOfInterval } from "date-fns";
import { exportToCsv } from '@/lib/utils';
import { Invoice } from '@/components/orders/invoice';
import { Checkbox } from '@/components/ui/checkbox';
import { ImportOrdersDialog } from '@/components/orders/import-orders-dialog';


type OrderStatus = 'New' | 'In Progress' | 'Ready for Delivery' | 'Delivered' | 'Completed' | 'Cancelled';
const orderStatuses: OrderStatus[] = ['New', 'In Progress', 'Ready for Delivery', 'Delivered', 'Completed', 'Cancelled'];

type ProductionStep = 'cutting' | 'stitching_start' | 'stitching_end';


export default function OrdersPage() {
  const [orders, setOrders] = useState<Order[]>([]);
  const [customers, setCustomers] = useState<Customer[]>([]);
  const [tailoringItems, setTailoringItems] = useState<TailoringItem[]>([]);
  const [artisans, setArtisans] = useState<Employee[]>([]);
  const [cuttingMasters, setCuttingMasters] = useState<Employee[]>([]);
  const [shopProfile, setShopProfile] = useState<ShopProfile | null>(null);
  const [isOrderDialogOpen, setIsOrderDialogOpen] = useState(false);
  const [isImportDialogOpen, setIsImportDialogOpen] = useState(false);
  const [editingOrder, setEditingOrder] = useState<Order | null>(null);
  const [deletingOrder, setDeletingOrder] = useState<Order | null>(null);
  const [printingOrder, setPrintingOrder] = useState<Order | null>(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilters, setStatusFilters] = useState<OrderStatus[]>([]);
  const [selectedMonth, setSelectedMonth] = useState<Date | null>(null);
  const [selectedOrders, setSelectedOrders] = useState<Set<string>>(new Set());


  useEffect(() => {
    loadData();

    const handleStorageChange = () => loadData();
    window.addEventListener('storage', handleStorageChange);
    return () => window.removeEventListener('storage', handleStorageChange);
  }, []);

  const loadData = () => {
    setOrders(getMockOrders());
    setCustomers(getMockCustomers());
    setTailoringItems(getMockTailoringItems());
    setArtisans(getArtisans());
    setCuttingMasters(getCuttingMasters());
    setShopProfile(getShopProfile());
  };

  const updateOrders = (newOrders: Order[]) => {
    localStorage.setItem('mockOrders', JSON.stringify(newOrders));
    loadData();
    setSelectedOrders(new Set()); 
  };
  
  const updateCustomers = (newCustomers: Customer[]) => {
      localStorage.setItem('mockCustomers', JSON.stringify(newCustomers));
      loadData();
  };


  const handleStatusChange = (orderId: string, newStatus: OrderStatus) => {
    const newOrders = orders.map(order => 
      order.id === orderId ? { ...order, status: newStatus } : order
    );
    updateOrders(newOrders);
  };
  
  const handleProductionStepChange = (orderId: string, step: ProductionStep) => {
    const today = format(new Date(), 'yyyy-MM-dd');
    const newOrders = orders.map(order => {
      if (order.id === orderId) {
        const updatedOrder = { ...order };
        if (step === 'cutting') {
          updatedOrder.cuttingCompletedDate = today;
        } else if (step === 'stitching_start') {
          updatedOrder.artisanAssignedDate = today;
        } else if (step === 'stitching_end') {
          updatedOrder.artisanCompletedDate = today;
        }
        return updatedOrder;
      }
      return order;
    });
    updateOrders(newOrders);
  };
  
  const handleArtisanChange = (orderId: string, artisanName: string) => {
      const newOrders = orders.map(order => 
          order.id === orderId ? { ...order, artisan: artisanName } : order
      );
      updateOrders(newOrders);
  };

  const handleCuttingMasterChange = (orderId: string, masterName: string) => {
      const newOrders = orders.map(order => 
          order.id === orderId ? { ...order, cuttingMaster: masterName } : order
      );
      updateOrders(newOrders);
  };

  const saveOrUpdateMeasurementsFromOrder = (customerId: string, orderItems: Order['items']) => {
    const customerMeasurements = getMockMeasurements(customerId);
    
    orderItems.forEach(item => {
      if (item.measurements) {
        const existingMeasurement = customerMeasurements.find(m => m.itemName === item.name);
        if (existingMeasurement) {
          if (existingMeasurement.measurements !== item.measurements) {
            updateMockMeasurement({ ...existingMeasurement, measurements: item.measurements });
          }
        } else {
          addMockMeasurement({
            customerId: customerId,
            itemName: item.name,
            measurements: item.measurements,
          });
        }
      }
    });
  };

  const handleAddOrder = (newOrderData: Omit<Order, 'id' | 'date'>) => {
    let customerId = newOrderData.customerId;
    const currentCustomers = getMockCustomers();
    if (!customerId) {
        const existingCustomer = currentCustomers.find(c => c.name.toLowerCase() === newOrderData.customer.toLowerCase() && c.phone === newOrderData.phone);
        if (existingCustomer) {
            customerId = existingCustomer.id;
        } else {
            const newCustomer: Customer = {
                id: `CUST${(currentCustomers.length + 1).toString().padStart(3, '0')}`,
                name: newOrderData.customer,
                phone: newOrderData.phone,
            };
            updateCustomers([newCustomer, ...currentCustomers]);
            customerId = newCustomer.id;
        }
    }
    
    const newOrder: Order = {
      id: `ORD${(orders.length + 1).toString().padStart(3, '0')}`,
      ...newOrderData,
      customerId: customerId,
      date: new Date().toISOString().split('T')[0],
    };
    
    updateOrders([newOrder, ...orders]);
    saveOrUpdateMeasurementsFromOrder(customerId, newOrder.items);
  };

  const handleEditOrder = (updatedOrderData: Order) => {
    const newOrders = orders.map(order => order.id === updatedOrderData.id ? updatedOrderData : order);
    updateOrders(newOrders);
    setEditingOrder(null);
    if (updatedOrderData.customerId) {
        saveOrUpdateMeasurementsFromOrder(updatedOrderData.customerId, updatedOrderData.items);
    }
  }

  const handleDeleteOrder = () => {
    if (deletingOrder) {
      if (deletingOrder.id === 'bulk-delete') {
          const newOrders = orders.filter(order => !selectedOrders.has(order.id));
          updateOrders(newOrders);
      } else {
          const newOrders = orders.filter(order => order.id !== deletingOrder.id);
          updateOrders(newOrders);
      }
      setDeletingOrder(null);
    }
  };

  const handleImport = (newItems: Order[]) => {
        const updatedOrders = [...orders];
        newItems.forEach(newItem => {
            const existingIndex = updatedOrders.findIndex(i => i.id === newItem.id);
            if (existingIndex > -1) {
                updatedOrders[existingIndex] = newItem;
            } else {
                updatedOrders.push(newItem);
            }
        });
        updateOrders(updatedOrders);
        setIsImportDialogOpen(false);
    };

  const openAddDialog = () => {
    setEditingOrder(null);
    setIsOrderDialogOpen(true);
  }

  const openEditDialog = (order: Order) => {
    setEditingOrder(order);
    setIsOrderDialogOpen(true);
  }
  
  const openDeleteDialog = (order: Order) => {
    setDeletingOrder(order);
  }

  const openBulkDeleteDialog = () => {
    setDeletingOrder({ id: 'bulk-delete', customer: `${selectedOrders.size} orders` } as any);
  }


  const closeDialogs = () => {
    setIsOrderDialogOpen(false);
    setEditingOrder(null);
    setDeletingOrder(null);
  }

  const handleExport = () => {
    const exportableData = filteredOrders.map(order => ({
        id: order.id,
        memoId: order.memoId,
        customer: order.customer,
        phone: order.phone,
        status: order.status,
        date: order.date,
        deliveryDate: order.deliveryDate,
        totalAmount: order.totalAmount,
        paidAmount: order.paidAmount,
        dueAmount: order.totalAmount - order.paidAmount,
        items: order.items.map(i => `${i.name} (x${i.quantity})`).join(', '),
        artisan: order.artisan,
    }));
    exportToCsv(exportableData, 'orders.csv');
  };

  const handlePrintInvoice = (order: Order) => {
    setPrintingOrder(order);
    setTimeout(() => {
      window.print();
    }, 100);
  }

  const handlePrintComplete = () => {
    setPrintingOrder(null);
  }

  const getStatusBadgeVariant = (status: OrderStatus) => {
    switch (status) {
        case "Completed": return "success" as const;
        case "In Progress":
        case "Ready for Delivery": return "secondary" as const;
        case "Delivered": return "accent" as const;
        case "Cancelled": return "destructive" as const;
        default: return "outline" as const;
    }
  }

  const getProductionStatus = (order: Order) => {
    if (order.artisanCompletedDate) return { text: 'Stitching Done', variant: 'success' as const, date: order.artisanCompletedDate};
    if (order.artisanAssignedDate) return { text: 'Stitching', variant: 'secondary' as const, date: order.artisanAssignedDate};
    if (order.cuttingCompletedDate) return { text: 'Cutting Done', variant: 'secondary' as const, date: order.cuttingCompletedDate};
    return { text: 'Pending', variant: 'outline' as const, date: null};
  }

  const handleStatusFilterChange = (status: OrderStatus) => {
    setStatusFilters(prev => 
      prev.includes(status) 
        ? prev.filter(s => s !== status) 
        : [...prev, status]
    );
  };
  
  const monthOptions = eachMonthOfInterval({
    start: subMonths(new Date(), 11),
    end: new Date(),
  }).reverse();

  const filteredOrders = orders.filter(order => {
    const search = searchTerm.toLowerCase();
    const matchesSearch =
        order.customer.toLowerCase().includes(search) ||
        order.phone.includes(search) ||
        order.id.toLowerCase().includes(search) ||
        (order.memoId && order.memoId.toLowerCase().includes(search));

    const matchesStatus =
        statusFilters.length === 0 || statusFilters.includes(order.status);

    const matchesMonth =
        !selectedMonth ||
        (new Date(order.date) >= startOfMonth(selectedMonth) && new Date(order.date) <= endOfMonth(selectedMonth));
    
    return matchesSearch && matchesStatus && matchesMonth;
  });

  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedOrders(new Set(filteredOrders.map(order => order.id)));
    } else {
      setSelectedOrders(new Set());
    }
  };

  const handleSelectItem = (orderId: string, checked: boolean) => {
    const newSelection = new Set(selectedOrders);
    if (checked) {
      newSelection.add(orderId);
    } else {
      newSelection.delete(orderId);
    }
    setSelectedOrders(newSelection);
  };

  if (printingOrder && shopProfile) {
    return <Invoice order={printingOrder} profile={shopProfile} onPrintComplete={handlePrintComplete} />;
  }

  return (
    <div className="flex flex-col gap-4">
      <PageHeader title="Orders" actionButtonText="Add Order" onActionButtonClick={openAddDialog}>
           <div className="relative w-full max-w-sm">
                <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
                <Input
                    type="search"
                    placeholder="Search by name, phone, order ID..."
                    className="pl-8 w-full"
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                />
            </div>
        <div className="flex items-center gap-2">
            {selectedOrders.size > 0 && (
                <Button size="sm" variant="destructive" className="h-8 gap-1" onClick={openBulkDeleteDialog}>
                    <Trash2 className="h-3.5 w-3.5" />
                    <span>Delete ({selectedOrders.size})</span>
                </Button>
            )}
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="outline" size="sm" className="h-8 gap-1">
                  <span>{selectedMonth ? format(selectedMonth, 'MMMM yyyy') : 'Filter by month'}</span>
                  <ChevronDown className="h-3.5 w-3.5" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end">
                <DropdownMenuItem onSelect={() => setSelectedMonth(null)}>All Months</DropdownMenuItem>
                {monthOptions.map((month) => (
                  <DropdownMenuItem key={month.toString()} onSelect={() => setSelectedMonth(month)}>
                    {format(month, 'MMMM yyyy')}
                  </DropdownMenuItem>
                ))}
              </DropdownMenuContent>
            </DropdownMenu>
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="outline" size="sm" className="h-8 gap-1">
                <ListFilter className="h-3.5 w-3.5" />
                <span className="sr-only sm:not-sr-only sm:whitespace-nowrap">
                  Filter
                </span>
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end">
              <DropdownMenuLabel>Filter by status</DropdownMenuLabel>
              <DropdownMenuSeparator />
              {orderStatuses.map(status => (
                  <DropdownMenuCheckboxItem
                    key={status}
                    checked={statusFilters.includes(status)}
                    onSelect={(event) => event.preventDefault()}
                    onCheckedChange={() => handleStatusFilterChange(status)}
                  >
                    {status}
                  </DropdownMenuCheckboxItem>
              ))}
            </DropdownMenuContent>
          </DropdownMenu>
          <Button size="sm" variant="outline" className="h-8 gap-1" onClick={() => setIsImportDialogOpen(true)}>
            <Upload className="h-3.5 w-3.5" />
            <span className="sr-only sm:not-sr-only sm:whitespace-nowrap">
              Import
            </span>
          </Button>
          <Button size="sm" variant="outline" className="h-8 gap-1" onClick={handleExport}>
            <File className="h-3.5 w-3.5" />
            <span className="sr-only sm:not-sr-only sm:whitespace-nowrap">
              Export
            </span>
          </Button>
        </div>
      </PageHeader>
      <OrderDialog 
        isOpen={isOrderDialogOpen}
        onOpenChange={setIsOrderDialogOpen}
        order={editingOrder}
        customers={customers}
        tailoringItems={tailoringItems}
        artisans={artisans}
        cuttingMasters={cuttingMasters}
        onOrderSubmit={(data) => {
          if (editingOrder) {
            handleEditOrder({ ...editingOrder, ...data });
          } else {
            handleAddOrder(data);
          }
          closeDialogs();
        }}
      />
       <ImportOrdersDialog
            isOpen={isImportDialogOpen}
            onOpenChange={setIsImportDialogOpen}
            onImport={handleImport}
        />
      <AlertDialog open={!!deletingOrder} onOpenChange={() => setDeletingOrder(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Are you absolutely sure?</AlertDialogTitle>
            <AlertDialogDescription>
              This action cannot be undone. This will permanently delete {deletingOrder?.id === 'bulk-delete' ? `the selected ${deletingOrder.customer}` : `order ${deletingOrder?.id}`} and remove its data from our servers.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction onClick={handleDeleteOrder}>Delete</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      <Tabs defaultValue="all">
        <TabsContent value="all">
          <Card>
            <CardHeader className="px-4 pt-4 sm:px-6 sm:pt-6">
              <CardTitle>All Orders</CardTitle>
              <CardDescription>
                Manage your customer orders and view their details.
              </CardDescription>
            </CardHeader>
            <CardContent className="p-0">
              <Table>
                <TableHeader>
                  <TableRow>
                     <TableHead className="w-[40px] pl-4">
                      <Checkbox
                        checked={selectedOrders.size > 0 && selectedOrders.size === filteredOrders.length}
                        onCheckedChange={(checked) => handleSelectAll(!!checked)}
                        aria-label="Select all"
                      />
                    </TableHead>
                    <TableHead className="hidden sm:table-cell w-[120px]">
                      Order / Memo ID
                    </TableHead>
                    <TableHead>Customer</TableHead>
                    <TableHead className="hidden md:table-cell">Items</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead className="hidden lg:table-cell">Artisan</TableHead>
                    <TableHead className="hidden md:table-cell">Delivery</TableHead>
                    <TableHead className="hidden xl:table-cell">Production</TableHead>
                    <TableHead className="text-right">Total / Due</TableHead>
                    <TableHead className="w-[40px]">
                      <span className="sr-only">Actions</span>
                    </TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredOrders.map((order) => {
                    const totalAmount = order.totalAmount || 0;
                    const paidAmount = order.paidAmount || 0;
                    const dueAmount = totalAmount - paidAmount;
                    const productionStatus = getProductionStatus(order);
                    const firstItem = order.items[0];
                    const remainingItemsCount = order.items.length - 1;
                    const isSelected = selectedOrders.has(order.id);

                    return (
                        <TableRow key={order.id} data-state={isSelected ? 'selected' : ''}>
                          <TableCell className="pl-4">
                              <Checkbox
                                  checked={isSelected}
                                  onCheckedChange={(checked) => handleSelectItem(order.id, !!checked)}
                                  aria-label={`Select order ${order.id}`}
                              />
                          </TableCell>
                        <TableCell className="hidden sm:table-cell font-medium">
                            <div>{order.id}</div>
                            {order.memoId && <div className="text-xs text-muted-foreground">M: {order.memoId}</div>}
                        </TableCell>
                        <TableCell>
                            <div className="font-medium">{order.customer}</div>
                            <div className="text-sm text-muted-foreground md:hidden">{order.phone}</div>
                        </TableCell>
                         <TableCell className="hidden md:table-cell">
                            <div className="font-medium">{firstItem?.name} (x{firstItem?.quantity})</div>
                            {remainingItemsCount > 0 && (
                                <div className="text-sm text-muted-foreground">
                                    + {remainingItemsCount} more item(s)
                                </div>
                            )}
                        </TableCell>
                        <TableCell>
                             <Badge variant={getStatusBadgeVariant(order.status)}>{order.status}</Badge>
                        </TableCell>
                        <TableCell className="hidden lg:table-cell">
                           <div className="font-medium">{order.artisan || 'N/A'}</div>
                           <div className="text-xs text-muted-foreground">Cutter: {order.cuttingMaster || 'N/A'}</div>
                        </TableCell>
                        <TableCell className="hidden md:table-cell">{order.deliveryDate || 'N/A'}</TableCell>
                        <TableCell className="hidden xl:table-cell">
                           <TooltipProvider>
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <Badge variant={productionStatus.variant} className="pointer-events-none">{productionStatus.text}</Badge>
                                </TooltipTrigger>
                                {productionStatus.date && <TooltipContent><p>{new Date(productionStatus.date).toLocaleDateString()}</p></TooltipContent>}
                              </Tooltip>
                            </TooltipProvider>
                        </TableCell>
                        <TableCell className="text-right font-medium">
                            <div>৳{totalAmount.toFixed(2)}</div>
                            {dueAmount > 0 && (
                                <div className="text-xs text-red-600">Due: ৳{dueAmount.toFixed(2)}</div>
                            )}
                        </TableCell>
                        <TableCell className="text-right">
                            <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                    <Button
                                    variant="ghost"
                                    className="h-8 w-8 p-0"
                                    >
                                    <span className="sr-only">
                                        Open menu
                                    </span>
                                    <MoreHorizontal className="h-4 w-4" />
                                    </Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent align="end">
                                    <DropdownMenuLabel>
                                    Actions
                                    </DropdownMenuLabel>
                                    <DropdownMenuItem onClick={() => openEditDialog(order)}>
                                    Edit
                                    </DropdownMenuItem>
                                    <DropdownMenuItem onClick={() => handlePrintInvoice(order)}>
                                        <Printer className="mr-2 h-4 w-4" /> Print Invoice
                                    </DropdownMenuItem>
                                     <DropdownMenuSeparator />
                                    <DropdownMenuLabel>Change Status</DropdownMenuLabel>
                                    {orderStatuses.map(status => (
                                        <DropdownMenuItem 
                                            key={status} 
                                            onSelect={() => handleStatusChange(order.id, status)}
                                            disabled={order.status === status}
                                        >
                                            {status}
                                        </DropdownMenuItem>
                                    ))}
                                    <DropdownMenuSeparator />
                                    <DropdownMenuSub>
                                        <DropdownMenuSubTrigger>Update Production</DropdownMenuSubTrigger>
                                        <DropdownMenuSubContent>
                                            <DropdownMenuItem onSelect={() => handleProductionStepChange(order.id, 'cutting')} disabled={!!order.cuttingCompletedDate}>Cutting Done</DropdownMenuItem>
                                            <DropdownMenuItem onSelect={() => handleProductionStepChange(order.id, 'stitching_start')} disabled={!!order.artisanAssignedDate}>Stitching Start</DropdownMenuItem>
                                            <DropdownMenuItem onSelect={() => handleProductionStepChange(order.id, 'stitching_end')} disabled={!!order.artisanCompletedDate}>Stitching End</DropdownMenuItem>
                                        </DropdownMenuSubContent>
                                    </DropdownMenuSub>
                                    <DropdownMenuSub>
                                        <DropdownMenuSubTrigger>Assign Artisan</DropdownMenuSubTrigger>
                                        <DropdownMenuSubContent>
                                            {artisans.map(artisan => (
                                                <DropdownMenuItem key={artisan.id} onSelect={() => handleArtisanChange(order.id, artisan.name)}>{artisan.name}</DropdownMenuItem>
                                            ))}
                                        </DropdownMenuSubContent>
                                    </DropdownMenuSub>
                                    <DropdownMenuSub>
                                        <DropdownMenuSubTrigger>Assign Cutter</DropdownMenuSubTrigger>
                                        <DropdownMenuSubContent>
                                            {cuttingMasters.map(master => (
                                                <DropdownMenuItem key={master.id} onSelect={() => handleCuttingMasterChange(order.id, master.name)}>{master.name}</DropdownMenuItem>
                                            ))}
                                        </DropdownMenuSubContent>
                                    </DropdownMenuSub>
                                    <DropdownMenuSeparator />
                                    <DropdownMenuItem className="text-red-600" onClick={() => openDeleteDialog(order)}>
                                    Delete Order
                                    </DropdownMenuItem>
                                </DropdownMenuContent>
                            </DropdownMenu>
                        </TableCell>
                        </TableRow>
                    )
                  })}
                </TableBody>
              </Table>
            </CardContent>
            <CardFooter>
              <div className="text-xs text-muted-foreground">
                Showing <strong>1-{filteredOrders.length}</strong> of <strong>{orders.length}</strong> orders
              </div>
            </CardFooter>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
}
