'use client';

import React, { useState, useTransition } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { calculateProfit, type ProfitCalculationInput, type ProfitCalculationOutput } from '@/ai/flows/profit-calculation';
import { CalculatorForm } from '@/components/profit-calculator/calculator-form';
import { ProfitResultCard } from '@/components/profit-calculator/profit-result-card';
import { useToast } from '@/hooks/use-toast';
import { PageHeader } from '@/components/page-header';
import { Skeleton } from '@/components/ui/skeleton';

export default function ProfitCalculatorPage() {
  const [result, setResult] = useState<ProfitCalculationOutput | null>(null);
  const [isPending, startTransition] = useTransition();
  const { toast } = useToast();

  const handleCalculate = (data: ProfitCalculationInput) => {
    startTransition(async () => {
      setResult(null); // Clear previous results
      const response = await calculateProfit(data);
      if (response) {
        setResult(response);
      } else {
        toast({
          variant: "destructive",
          title: "Error",
          description: "Failed to calculate profit. Please try again.",
        })
      }
    });
  };

  return (
    <div className="flex flex-col gap-4">
        <PageHeader title="AI Profit Calculator" />
        <div className="grid md:grid-cols-2 gap-8 items-start">
        <Card>
            <CardHeader>
            <CardTitle className="font-headline">Enter Financials</CardTitle>
            <CardDescription>Provide your total income and expenses to calculate your net profit and get AI-powered suggestions.</CardDescription>
            </CardHeader>
            <CardContent>
            <CalculatorForm onSubmit={handleCalculate} isSubmitting={isPending} />
            </CardContent>
        </Card>
        <div>
            {isPending ? (
            <div>
                <Card>
                    <CardHeader>
                        <Skeleton className="h-6 w-1/2" />
                        <Skeleton className="h-4 w-3/4" />
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <div className="grid grid-cols-2 gap-4">
                            <Skeleton className="h-20 w-full" />
                            <Skeleton className="h-20 w-full" />
                        </div>
                        <div className="space-y-2">
                            <Skeleton className="h-5 w-1/3" />
                            <Skeleton className="h-4 w-full" />
                            <Skeleton className="h-4 w-full" />
                            <Skeleton className="h-4 w-4/5" />
                        </div>
                    </CardContent>
                </Card>
            </div>
            ) : result ? (
            <div>
                <ProfitResultCard result={result} />
            </div>
            ) : (
            <div className="flex items-center justify-center h-full">
                <Card className="w-full text-center">
                    <CardHeader>
                        <CardTitle className="font-headline">Awaiting Calculation</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <p className="text-muted-foreground">Your profit analysis will appear here.</p>
                    </CardContent>
                </Card>
            </div>
            )}
        </div>
        </div>
    </div>
  );
}
