'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Textarea } from '../ui/textarea';
import type { CustomerMeasurement, TailoringItem } from '@/lib/data';
import { DialogFooter } from '../ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../ui/select';

const formSchema = z.object({
  itemName: z.string().min(2, { message: "Item name must be at least 2 characters." }),
  measurements: z.string().min(5, { message: "Measurements must be at least 5 characters." }),
});

export type MeasurementFormValues = z.infer<typeof formSchema>;

type MeasurementFormProps = {
  measurement: CustomerMeasurement | null;
  onSubmit: (data: MeasurementFormValues) => void;
  tailoringItems: TailoringItem[];
};

export function MeasurementForm({ measurement, onSubmit, tailoringItems }: MeasurementFormProps) {
  const isEditMode = !!measurement;

  const form = useForm<MeasurementFormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: isEditMode ? {
      itemName: measurement.itemName,
      measurements: measurement.measurements,
    } : {
      itemName: '',
      measurements: '',
    },
  });

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4 py-4">
        <FormField
          control={form.control}
          name="itemName"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Item Name</FormLabel>
                <Select onValueChange={field.onChange} defaultValue={field.value}>
                    <FormControl>
                        <SelectTrigger>
                            <SelectValue placeholder="Select an item" />
                        </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                        {tailoringItems.map(item => (
                            <SelectItem key={item.id} value={item.name}>
                                {item.name}
                            </SelectItem>
                        ))}
                    </SelectContent>
                </Select>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="measurements"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Measurements</FormLabel>
              <FormControl>
                <Textarea placeholder="e.g., Length: 28, Chest: 42, Sleeve: 24" className="min-h-[100px]" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <DialogFooter>
            <Button type="button" variant="outline" onClick={() => form.reset()}>Cancel</Button>
            <Button type="submit">{isEditMode ? 'Save Changes' : 'Add Measurement'}</Button>
        </DialogFooter>
      </form>
    </Form>
  );
}
