'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { DialogFooter } from '../ui/dialog';
import type { InventoryItem } from '@/lib/data';

const inventoryTypes = ['Cotton', 'Silk', 'Wool', 'Linen'] as const;

const formSchema = z.object({
  name: z.string().min(2, { message: 'Name must be at least 2 characters.' }),
  type: z.enum(inventoryTypes),
  color: z.string().min(2, { message: 'Color must be at least 2 characters.' }),
  quantity: z.coerce.number().min(0, { message: 'Quantity must be a positive number.' }),
  pricePerMeter: z.coerce.number().min(0.01, { message: 'Price must be a positive number.' }),
  lowStockThreshold: z.coerce.number().min(0, { message: 'Threshold must be a positive number.' }),
});

export type InventoryFormValues = z.infer<typeof formSchema>;

type InventoryFormProps = {
  item: InventoryItem | null;
  onSubmit: (data: InventoryFormValues) => void;
};

export function InventoryForm({ item, onSubmit }: InventoryFormProps) {
  const isEditMode = !!item;

  const form = useForm<InventoryFormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: isEditMode ? {
        ...item
    } : {
      name: '',
      type: 'Cotton',
      color: '',
      quantity: 0,
      pricePerMeter: 0,
      lowStockThreshold: 10,
    },
  });

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4 py-4">
        <FormField
          control={form.control}
          name="name"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Material Name</FormLabel>
              <FormControl><Input placeholder="e.g., Royal Blue Silk" {...field} /></FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <div className="grid grid-cols-2 gap-4">
            <FormField
                control={form.control}
                name="type"
                render={({ field }) => (
                    <FormItem>
                        <FormLabel>Type</FormLabel>
                        <Select onValueChange={field.onChange} defaultValue={field.value}>
                            <FormControl><SelectTrigger><SelectValue placeholder="Select type" /></SelectTrigger></FormControl>
                            <SelectContent>
                                {inventoryTypes.map(type => (
                                    <SelectItem key={type} value={type}>{type}</SelectItem>
                                ))}
                            </SelectContent>
                        </Select>
                        <FormMessage />
                    </FormItem>
                )}
            />
            <FormField
                control={form.control}
                name="color"
                render={({ field }) => (
                    <FormItem>
                        <FormLabel>Color</FormLabel>
                        <FormControl><Input placeholder="e.g., Blue" {...field} /></FormControl>
                        <FormMessage />
                    </FormItem>
                )}
            />
        </div>
         <div className="grid grid-cols-2 gap-4">
            <FormField
                control={form.control}
                name="quantity"
                render={({ field }) => (
                    <FormItem>
                        <FormLabel>Quantity (meters)</FormLabel>
                        <FormControl><Input type="number" placeholder="e.g., 50" {...field} /></FormControl>
                        <FormMessage />
                    </FormItem>
                )}
            />
            <FormField
                control={form.control}
                name="pricePerMeter"
                render={({ field }) => (
                    <FormItem>
                        <FormLabel>Price per Meter</FormLabel>
                        <FormControl><Input type="number" step="0.01" placeholder="e.g., 25.00" {...field} /></FormControl>
                        <FormMessage />
                    </FormItem>
                )}
            />
        </div>
        <FormField
            control={form.control}
            name="lowStockThreshold"
            render={({ field }) => (
                <FormItem>
                    <FormLabel>Low Stock Threshold</FormLabel>
                    <FormControl><Input type="number" placeholder="e.g., 10" {...field} /></FormControl>
                    <FormMessage />
                </FormItem>
            )}
        />
        <DialogFooter>
          <Button type="button" variant="outline" onClick={() => form.reset()}>Cancel</Button>
          <Button type="submit">{isEditMode ? 'Save Changes' : 'Add Material'}</Button>
        </DialogFooter>
      </form>
    </Form>
  );
}
