'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { z } from 'zod';

import { Button } from "@/components/ui/button";
import {
  DialogFooter
} from '@/components/ui/dialog';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import type { TailoringItem } from '@/lib/data';

const itemSchema = z.object({
  name: z.string().min(2, { message: "Item name must be at least 2 characters." }),
  category: z.enum(['Men', 'Women', 'Kids', 'Other']),
  sewingPrice: z.coerce.number().min(0, { message: "Sewing price must be a positive number." }),
  artisanWage: z.coerce.number().min(0, { message: "Artisan wage must be a positive number." }),
});

export type ItemFormValues = z.infer<typeof itemSchema>;

type ItemFormProps = {
  item: TailoringItem | null;
  onSubmit: (data: ItemFormValues) => void;
  onCancel: () => void;
};


export function ItemForm({ item, onSubmit, onCancel }: ItemFormProps) {
    const isEditMode = !!item;
    
    const form = useForm<ItemFormValues>({
        resolver: zodResolver(itemSchema),
        defaultValues: item ? {
            name: item.name,
            category: item.category,
            sewingPrice: item.sewingPrice,
            artisanWage: item.artisanWage,
        } : {
            name: '',
            category: 'Men',
            sewingPrice: 0,
            artisanWage: 0,
        }
    });

    return (
        <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4 py-4">
                <FormField
                    control={form.control}
                    name="name"
                    render={({ field }) => (
                        <FormItem>
                            <FormLabel>Item Name</FormLabel>
                            <FormControl><Input placeholder="e.g., Pant" {...field} /></FormControl>
                            <FormMessage />
                        </FormItem>
                    )}
                />
                <FormField
                    control={form.control}
                    name="category"
                    render={({ field }) => (
                        <FormItem>
                            <FormLabel>Category</FormLabel>
                            <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl><SelectTrigger><SelectValue placeholder="Select a category" /></SelectTrigger></FormControl>
                                <SelectContent>
                                    <SelectItem value="Men">Men</SelectItem>
                                    <SelectItem value="Women">Women</SelectItem>
                                    <SelectItem value="Kids">Kids</SelectItem>
                                    <SelectItem value="Other">Other</SelectItem>
                                </SelectContent>
                            </Select>
                            <FormMessage />
                        </FormItem>
                    )}
                />
                <FormField
                    control={form.control}
                    name="sewingPrice"
                    render={({ field }) => (
                        <FormItem>
                            <FormLabel>Sewing Price</FormLabel>
                            <FormControl><Input type="number" placeholder="e.g., 800" {...field} /></FormControl>
                            <FormMessage />
                        </FormItem>
                    )}
                />
                 <FormField
                    control={form.control}
                    name="artisanWage"
                    render={({ field }) => (
                        <FormItem>
                            <FormLabel>Artisan Wage</FormLabel>
                            <FormControl><Input type="number" placeholder="e.g., 250" {...field} /></FormControl>
                            <FormMessage />
                        </FormItem>
                    )}
                />
                <DialogFooter>
                    <Button type="button" variant="outline" onClick={onCancel}>Cancel</Button>
                    <Button type="submit">{isEditMode ? 'Save Changes' : 'Add Item'}</Button>
                </DialogFooter>
            </form>
        </Form>
    );
}

