'use client';

import { useEffect, useState } from 'react';
import { zodResolver } from '@hookform/resolvers/zod';
import { useForm, useFieldArray, Controller } from 'react-hook-form';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import type { Order, Customer, Employee, TailoringItem, CustomerMeasurement } from '@/lib/data';
import { getMockMeasurements } from '@/lib/data';
import { Popover, PopoverContent, PopoverTrigger } from '../ui/popover';
import { cn } from '@/lib/utils';
import { Check, ChevronsUpDown, PlusCircle, Trash2 } from 'lucide-react';
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from '../ui/command';
import { DialogFooter } from '../ui/dialog';
import { Separator } from '../ui/separator';

const orderItemSchema = z.object({
  id: z.string(),
  name: z.string().min(1, { message: 'Item name is required.' }),
  quantity: z.coerce.number().min(1, { message: 'Quantity must be at least 1.' }),
  price: z.coerce.number().min(0, { message: 'Price must be a positive number.' }),
  artisanWage: z.coerce.number().min(0, { message: 'Artisan wage must be a positive number.' }),
  measurements: z.string().optional(),
});

const formSchema = z.object({
  customerId: z.string().optional(),
  memoId: z.string().optional(),
  customer: z.string().min(2, { message: 'Customer name must be at least 2 characters.' }),
  phone: z.string().min(10, { message: 'Phone number must be at least 10 digits.' }),
  items: z.array(orderItemSchema).min(1, { message: 'At least one item is required.' }),
  totalAmount: z.coerce.number(),
  totalArtisanWage: z.coerce.number(),
  paidAmount: z.coerce.number().min(0, { message: 'Paid amount must be a positive number.' }),
  type: z.enum(['pickup', 'delivery']),
  status: z.enum(['New', 'In Progress', 'Ready for Delivery', 'Delivered', 'Completed', 'Cancelled']),
  deliveryDate: z.string().optional(),
  artisan: z.string().optional(),
  cuttingMaster: z.string().optional(),
  artisanAssignedDate: z.string().optional(),
  artisanCompletedDate: z.string().optional(),
  cuttingCompletedDate: z.string().optional(),
});

type OrderFormValues = z.infer<typeof formSchema>;

type OrderFormProps = {
  order?: Order | null;
  customers: Customer[];
  tailoringItems: TailoringItem[];
  artisans: Employee[];
  cuttingMasters: Employee[];
  onSubmit: (data: Omit<Order, 'id' | 'date'>) => void;
};

export function OrderForm({ order, customers, tailoringItems, artisans, cuttingMasters, onSubmit }: OrderFormProps) {
  const isEditMode = !!order;
  const [customerPopoverOpen, setCustomerPopoverOpen] = useState(false);
  const [customerMeasurements, setCustomerMeasurements] = useState<CustomerMeasurement[]>([]);
  
  const form = useForm<OrderFormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: isEditMode && order ? {
        ...order,
        deliveryDate: order.deliveryDate ? order.deliveryDate.split('T')[0] : '',
        artisanAssignedDate: order.artisanAssignedDate ? order.artisanAssignedDate.split('T')[0] : '',
        artisanCompletedDate: order.artisanCompletedDate ? order.artisanCompletedDate.split('T')[0] : '',
        cuttingCompletedDate: order.cuttingCompletedDate ? order.cuttingCompletedDate.split('T')[0] : '',
    } : {
      customerId: '',
      memoId: '',
      customer: '',
      phone: '',
      items: [],
      totalAmount: 0,
      totalArtisanWage: 0,
      paidAmount: 0,
      type: 'pickup',
      status: 'New',
      deliveryDate: '',
      artisan: '',
      cuttingMaster: '',
      artisanAssignedDate: '',
      artisanCompletedDate: '',
      cuttingCompletedDate: '',
    },
  });

  const { fields, append, remove } = useFieldArray({
    control: form.control,
    name: "items"
  });
  
  const watchedItems = form.watch('items');
  const watchedCustomerId = form.watch('customerId');

  useEffect(() => {
    const total = watchedItems.reduce((sum, item) => sum + ((item.price || 0) * (item.quantity || 1)), 0);
    const totalWage = watchedItems.reduce((sum, item) => sum + ((item.artisanWage || 0) * (item.quantity || 1)), 0);
    form.setValue('totalAmount', total);
    form.setValue('totalArtisanWage', totalWage);
  }, [watchedItems, form]);


   useEffect(() => {
    if (order) {
      const resetData = {
        ...order,
        deliveryDate: order.deliveryDate ? order.deliveryDate.split('T')[0] : '',
        artisanAssignedDate: order.artisanAssignedDate ? order.artisanAssignedDate.split('T')[0] : '',
        artisanCompletedDate: order.artisanCompletedDate ? order.artisanCompletedDate.split('T')[0] : '',
        cuttingCompletedDate: order.cuttingCompletedDate ? order.cuttingCompletedDate.split('T')[0] : '',
      };
      form.reset(resetData);
       if (order.customerId) {
        setCustomerMeasurements(getMockMeasurements(order.customerId));
      }
    } else {
      form.reset({
        customerId: '',
        memoId: '',
        customer: '',
        phone: '',
        items: [],
        totalAmount: 0,
        totalArtisanWage: 0,
        paidAmount: 0,
        type: 'pickup',
        status: 'New',
        deliveryDate: '',
        artisan: '',
        cuttingMaster: '',
        artisanAssignedDate: '',
        artisanCompletedDate: '',
        cuttingCompletedDate: '',
      });
      setCustomerMeasurements([]);
    }
  }, [order, form]);
  
  useEffect(() => {
    if (watchedCustomerId) {
      const measurements = getMockMeasurements(watchedCustomerId);
      setCustomerMeasurements(measurements);
    } else {
      setCustomerMeasurements([]);
    }
  }, [watchedCustomerId]);
  
  const handleAddItem = () => {
    append({
        id: `item-${Date.now()}`,
        name: '',
        quantity: 1,
        price: 0,
        artisanWage: 0,
        measurements: '',
    });
  };

  const applySavedMeasurement = (itemIndex: number, itemName: string) => {
    const savedMeasurement = customerMeasurements.find(m => m.itemName === itemName);
    if (savedMeasurement) {
        form.setValue(`items.${itemIndex}.measurements`, savedMeasurement.measurements);
    }
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4 py-4 max-h-[80svh] sm:max-h-[70vh] overflow-y-auto pr-4">
        <div className="grid sm:grid-cols-2 gap-4">
            <FormItem className="flex flex-col">
            <FormLabel>Existing Customer</FormLabel>
                <Popover open={customerPopoverOpen} onOpenChange={setCustomerPopoverOpen}>
                    <PopoverTrigger asChild>
                        <FormControl>
                        <Button
                            variant="outline"
                            role="combobox"
                            className={cn(
                            "w-full justify-between",
                            !form.watch('customer') && "text-muted-foreground"
                            )}
                        >
                            {form.watch('customer')
                            ? customers.find(
                                (c) => c.id === form.watch('customerId')
                                )?.name
                            : "Select customer"}
                            <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                        </Button>
                        </FormControl>
                    </PopoverTrigger>
                    <PopoverContent className="w-[--radix-popover-trigger-width] p-0">
                        <Command>
                            <CommandInput placeholder="Search customer..." />
                            <CommandList>
                                <CommandEmpty>No customer found.</CommandEmpty>
                                <CommandGroup>
                                    {customers.map((customer) => (
                                    <CommandItem
                                        value={`${customer.name} - ${customer.phone}`}
                                        key={customer.id}
                                        onSelect={() => {
                                            form.setValue("customerId", customer.id)
                                            form.setValue("customer", customer.name)
                                            form.setValue("phone", customer.phone)
                                            setCustomerPopoverOpen(false)
                                        }}
                                    >
                                        <Check
                                        className={cn(
                                            "mr-2 h-4 w-4",
                                            customer.id === form.watch('customerId')
                                            ? "opacity-100"
                                            : "opacity-0"
                                        )}
                                        />
                                        {customer.name} - {customer.phone}
                                    </CommandItem>
                                    ))}
                                </CommandGroup>
                            </CommandList>
                        </Command>
                    </PopoverContent>
                </Popover>
                <FormMessage />
            </FormItem>
             <FormField
                control={form.control}
                name="memoId"
                render={({ field }) => (
                    <FormItem>
                    <FormLabel>Memo ID (Optional)</FormLabel>
                    <FormControl>
                        <Input placeholder="e.g., 101" {...field} />
                    </FormControl>
                    <FormMessage />
                    </FormItem>
                )}
            />
        </div>


        <div className="grid sm:grid-cols-2 gap-4">
            <FormField
            control={form.control}
            name="customer"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Customer Name</FormLabel>
                <FormControl>
                    <Input placeholder="e.g., John Doe" {...field} />
                </FormControl>
                <FormMessage />
                </FormItem>
            )}
            />
            <FormField
            control={form.control}
            name="phone"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Mobile Number</FormLabel>
                <FormControl>
                    <Input type="tel" placeholder="e.g., 123-456-7890" {...field} />
                </FormControl>
                <FormMessage />
                </FormItem>
            )}
            />
        </div>

        <Separator />

        <div>
            <h3 className="text-lg font-medium mb-2">Order Items</h3>
            <div className="space-y-4">
                {fields.map((field, index) => {
                    const currentItemName = form.watch(`items.${index}.name`);
                    const hasSavedMeasurement = customerMeasurements.some(m => m.itemName === currentItemName);
                    
                    return (
                    <div key={field.id} className="p-4 border rounded-lg space-y-4 relative">
                        <Button
                            type="button"
                            variant="ghost"
                            size="icon"
                            className="absolute top-2 right-2 h-6 w-6"
                            onClick={() => remove(index)}
                        >
                            <Trash2 className="h-4 w-4 text-red-500" />
                            <span className="sr-only">Remove Item</span>
                        </Button>
                        <div className="grid sm:grid-cols-2 gap-4">
                            <FormField
                                control={form.control}
                                name={`items.${index}.name`}
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Item</FormLabel>
                                        <Select onValueChange={(value) => {
                                            field.onChange(value);
                                            const selectedItem = tailoringItems.find(i => i.name === value);
                                            if(selectedItem) {
                                                form.setValue(`items.${index}.price`, selectedItem.sewingPrice);
                                                form.setValue(`items.${index}.artisanWage`, selectedItem.artisanWage || 0);
                                            }
                                        }} defaultValue={field.value}>
                                            <FormControl>
                                            <SelectTrigger>
                                                <SelectValue placeholder="Select an item" />
                                            </SelectTrigger>
                                            </FormControl>
                                            <SelectContent>
                                            {tailoringItems.map(item => (
                                                <SelectItem key={item.id} value={item.name}>
                                                {item.name}
                                                </SelectItem>
                                            ))}
                                            </SelectContent>
                                        </Select>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                             <FormField
                                control={form.control}
                                name={`items.${index}.quantity`}
                                render={({ field }) => (
                                    <FormItem>
                                    <FormLabel>Quantity</FormLabel>
                                    <FormControl>
                                        <Input type="number" placeholder="1" {...field} />
                                    </FormControl>
                                    <FormMessage />
                                    </FormItem>
                                )}
                            />
                        </div>
                         <FormField
                            control={form.control}
                            name={`items.${index}.measurements`}
                            render={({ field }) => (
                                <FormItem>
                                <div className="flex justify-between items-center">
                                    <FormLabel>Measurements</FormLabel>
                                    {hasSavedMeasurement && (
                                        <Button 
                                            type="button" 
                                            variant="link" 
                                            className="h-auto p-0"
                                            onClick={() => applySavedMeasurement(index, currentItemName)}
                                        >
                                            Apply Saved Measurement
                                        </Button>
                                    )}
                                </div>
                                <FormControl>
                                    <Textarea placeholder="e.g., Chest: 42, Waist: 34, Length: 30" {...field} />
                                </FormControl>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                        <div className="grid sm:grid-cols-2 gap-4">
                             <FormField
                                control={form.control}
                                name={`items.${index}.price`}
                                render={({ field }) => (
                                    <FormItem>
                                    <FormLabel>Sewing Price</FormLabel>
                                    <FormControl>
                                        <Input type="number" placeholder="e.g., 800" {...field} />
                                    </FormControl>
                                    <FormMessage />
                                    </FormItem>
                                )}
                            />
                            <FormField
                                control={form.control}
                                name={`items.${index}.artisanWage`}
                                render={({ field }) => (
                                    <FormItem>
                                    <FormLabel>Artisan Wage</FormLabel>
                                    <FormControl>
                                        <Input type="number" placeholder="e.g., 250" {...field} />
                                    </FormControl>
                                    <FormMessage />
                                    </FormItem>
                                )}
                            />
                        </div>
                    </div>
                )})}
                 {form.formState.errors.items && fields.length === 0 && (
                     <p className="text-sm font-medium text-destructive">{form.formState.errors.items.message}</p>
                 )}
            </div>
             <Button
                type="button"
                variant="outline"
                size="sm"
                className="mt-2"
                onClick={handleAddItem}
            >
                <PlusCircle className="mr-2 h-4 w-4" />
                Add Item
            </Button>
        </div>

        <Separator />

        <div className="grid sm:grid-cols-2 gap-4">
            <FormItem>
                <FormLabel>Total Amount</FormLabel>
                <FormControl>
                    <Input type="number" placeholder="0.00" value={form.watch('totalAmount')} readOnly className="bg-muted" />
                </FormControl>
                <FormMessage />
            </FormItem>
            <FormField
            control={form.control}
            name="paidAmount"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Paid Amount</FormLabel>
                <FormControl>
                    <Input type="number" placeholder="e.g., 100.00" {...field} />
                </FormControl>
                <FormMessage />
                </FormItem>
            )}
            />
        </div>
        <div className="grid sm:grid-cols-2 gap-4">
            <FormField
            control={form.control}
            name="status"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Status</FormLabel>
                <Select onValueChange={field.onChange} defaultValue={field.value}>
                    <FormControl>
                    <SelectTrigger>
                        <SelectValue placeholder="Select order status" />
                    </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                    <SelectItem value="New">New</SelectItem>
                    <SelectItem value="In Progress">In Progress</SelectItem>
                    <SelectItem value="Ready for Delivery">Ready for Delivery</SelectItem>
                    <SelectItem value="Delivered">Delivered</SelectItem>
                    <SelectItem value="Completed">Completed</SelectItem>
                    <SelectItem value="Cancelled">Cancelled</SelectItem>
                    </SelectContent>
                </Select>
                <FormMessage />
                </FormItem>
            )}
            />
            <FormField
            control={form.control}
            name="type"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Order Type</FormLabel>
                <Select onValueChange={field.onChange} defaultValue={field.value}>
                    <FormControl>
                    <SelectTrigger>
                        <SelectValue placeholder="Select order type" />
                    </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                    <SelectItem value="pickup">Pickup</SelectItem>
                    <SelectItem value="delivery">Delivery</SelectItem>
                    </SelectContent>
                </Select>
                <FormMessage />
                </FormItem>
            )}
            />
        </div>
        <FormField
            control={form.control}
            name="deliveryDate"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Delivery Date</FormLabel>
                <FormControl>
                    <Input type="date" {...field} />
                </FormControl>
                <FormMessage />
                </FormItem>
            )}
        />
        <div className="grid sm:grid-cols-2 gap-4">
            <FormField
                control={form.control}
                name="artisan"
                render={({ field }) => (
                    <FormItem>
                    <FormLabel>Assign Artisan</FormLabel>
                    <Select onValueChange={field.onChange} defaultValue={field.value}>
                        <FormControl>
                        <SelectTrigger>
                            <SelectValue placeholder="Select Artisan" />
                        </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                        {artisans.map(artisan => (
                            <SelectItem key={artisan.id} value={artisan.name}>
                            {artisan.name}
                            </SelectItem>
                        ))}
                        </SelectContent>
                    </Select>
                    <FormMessage />
                    </FormItem>
                )}
            />
            <FormField
                control={form.control}
                name="cuttingMaster"
                render={({ field }) => (
                    <FormItem>
                    <FormLabel>Assign Cutting Master</FormLabel>
                    <Select onValueChange={field.onChange} defaultValue={field.value}>
                        <FormControl>
                        <SelectTrigger>
                            <SelectValue placeholder="Select Cutting Master" />
                        </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                        {cuttingMasters.map(master => (
                            <SelectItem key={master.id} value={master.name}>
                            {master.name}
                            </SelectItem>
                        ))}
                        </SelectContent>
                    </Select>
                    <FormMessage />
                    </FormItem>
                )}
            />
        </div>
        <div className="grid sm:grid-cols-2 gap-4">
            <FormField
                control={form.control}
                name="artisanAssignedDate"
                render={({ field }) => (
                    <FormItem>
                    <FormLabel>Artisan Assigned</FormLabel>
                    <FormControl>
                        <Input type="date" {...field} />
                    </FormControl>
                    <FormMessage />
                    </FormItem>
                )}
            />
             <FormField
                control={form.control}
                name="cuttingCompletedDate"
                render={({ field }) => (
                    <FormItem>
                    <FormLabel>Cutting Completed</FormLabel>
                    <FormControl>
                        <Input type="date" {...field} />
                    </FormControl>
                    <FormMessage />
                    </FormItem>
                )}
            />
        </div>
         <FormField
            control={form.control}
            name="artisanCompletedDate"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Artisan Completed</FormLabel>
                <FormControl>
                    <Input type="date" {...field} />
                </FormControl>
                <FormMessage />
                </FormItem>
            )}
        />
        <DialogFooter>
          <Button type="submit" className="w-full sm:w-auto">{isEditMode ? 'Save Changes' : 'Create Order'}</Button>
        </DialogFooter>
      </form>
    </Form>
  );
}
