'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import type { ProfitCalculationInput } from '@/ai/flows/profit-calculation';

const formSchema = z.object({
  totalIncome: z.coerce.number().min(0, { message: 'Income must be a positive number.' }),
  totalExpenses: z.coerce.number().min(0, { message: 'Expenses must be a positive number.' }),
});

type CalculatorFormProps = {
  onSubmit: (data: ProfitCalculationInput) => void;
  isSubmitting: boolean;
};

export function CalculatorForm({ onSubmit, isSubmitting }: CalculatorFormProps) {
  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      totalIncome: 0,
      totalExpenses: 0,
    },
  });

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
        <FormField
          control={form.control}
          name="totalIncome"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Total Income</FormLabel>
              <FormControl>
                <Input type="number" placeholder="e.g., 5000" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="totalExpenses"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Total Expenses</FormLabel>
              <FormControl>
                <Input type="number" placeholder="e.g., 2500" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <Button type="submit" className="w-full" disabled={isSubmitting}>
          {isSubmitting ? 'Calculating...' : 'Calculate Profit'}
        </Button>
      </form>
    </Form>
  );
}
