// API Route: /api/auth/login
// Handles user authentication

import { NextApiRequest, NextApiResponse } from 'next';
import { userService } from '@/lib/database/mysql-service';
import { generateToken, verifyPassword } from '@/lib/middleware/auth';
import { withErrorHandler } from '@/lib/middleware/error-handler';
import { validateRequest } from '@/lib/middleware/validation';
import { z } from 'zod';

const loginSchema = z.object({
  email: z.string().email('Invalid email format'),
  password: z.string().min(1, 'Password is required'),
});

async function handler(req: NextApiRequest, res: NextApiResponse) {
  if (req.method !== 'POST') {
    res.setHeader('Allow', ['POST']);
    return res.status(405).json({ error: `Method ${req.method} not allowed` });
  }

  try {
    const { email, password } = loginSchema.parse(req.body);

    // Find user by email
    const users = await userService.findAll({ email: email.toLowerCase() });
    const user = users.data[0];

    if (!user) {
      return res.status(401).json({
        success: false,
        error: 'Invalid email or password',
        code: 'INVALID_CREDENTIALS'
      });
    }

    // Check if user is active
    if (!user.is_active) {
      return res.status(401).json({
        success: false,
        error: 'Account is deactivated',
        code: 'ACCOUNT_DEACTIVATED'
      });
    }

    // Verify password
    const isValidPassword = await verifyPassword(password, user.password_hash);
    if (!isValidPassword) {
      return res.status(401).json({
        success: false,
        error: 'Invalid email or password',
        code: 'INVALID_CREDENTIALS'
      });
    }

    // Update last login
    await userService.update(user.id, {
      last_login: new Date(),
    });

    // Generate JWT token
    const token = generateToken(user.id);

    // Return user data (without password)
    const { password_hash, ...userWithoutPassword } = user;

    return res.status(200).json({
      success: true,
      data: {
        user: userWithoutPassword,
        token,
      },
      message: 'Login successful',
    });

  } catch (error: any) {
    if (error.name === 'ZodError') {
      return res.status(400).json({
        success: false,
        error: 'Validation error',
        details: error.errors,
        code: 'VALIDATION_ERROR'
      });
    }

    throw error;
  }
}

export default withErrorHandler(handler);
