'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { z } from 'zod';
import { Button } from '@/components/ui/button';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Textarea } from '../ui/textarea';
import type { Employee, Expense } from '@/lib/data';
import { format } from 'date-fns';
import { DialogFooter } from '../ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../ui/select';
import { expenseCategories } from '@/lib/data';
import { useEffect } from 'react';

const formSchema = z.object({
  category: z.string().min(2, { message: 'Category must be at least 2 characters.' }),
  description: z.string().optional(),
  amount: z.coerce.number().min(0.01, { message: 'Amount must be greater than 0.' }),
  date: z.string().refine((val) => !isNaN(Date.parse(val)), { message: "Invalid date" }),
  person: z.string().optional(),
});

export type ExpenseFormValues = z.infer<typeof formSchema>;

type AddExpenseFormProps = {
  expense: Expense | null;
  employees: Employee[];
  onSubmit: (data: ExpenseFormValues) => void;
  defaultCategory?: string;
  defaultPerson?: string;
};

export function AddExpenseForm({ expense, employees, onSubmit, defaultCategory, defaultPerson }: AddExpenseFormProps) {
    const isEditMode = !!expense;
    const form = useForm<ExpenseFormValues>({
        resolver: zodResolver(formSchema),
        defaultValues: isEditMode && expense ? {
            ...expense,
             date: format(new Date(expense.date), 'yyyy-MM-dd'),
        } : {
        category: defaultCategory || '',
        description: '',
        amount: 0,
        date: format(new Date(), 'yyyy-MM-dd'),
        person: defaultPerson || '',
        },
  });
  
   const selectedCategory = form.watch('category');

  useEffect(() => {
    if (defaultCategory) {
      form.setValue('category', defaultCategory);
    }
    if (defaultPerson) {
      form.setValue('person', defaultPerson);
    }
  }, [defaultCategory, defaultPerson, form]);


  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4 py-4">
        <FormField
          control={form.control}
          name="category"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Category</FormLabel>
              <Select onValueChange={field.onChange} defaultValue={field.value} disabled={!!defaultCategory}>
                <FormControl>
                  <SelectTrigger>
                    <SelectValue placeholder="Select a category" />
                  </SelectTrigger>
                </FormControl>
                <SelectContent>
                  {expenseCategories.map(cat => (
                    <SelectItem key={cat} value={cat}>{cat}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <FormMessage />
            </FormItem>
          )}
        />
        
        {selectedCategory === 'Salary Advance' && (
            <FormField
              control={form.control}
              name="person"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Select Person</FormLabel>
                   <Select onValueChange={field.onChange} defaultValue={field.value} disabled={!!defaultPerson}>
                    <FormControl>
                      <SelectTrigger>
                        <SelectValue placeholder="Select an employee" />
                      </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                      {employees.map(employee => (
                        <SelectItem key={employee.id} value={employee.name}>{employee.name}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />
        )}

         <FormField
          control={form.control}
          name="amount"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Amount</FormLabel>
              <FormControl>
                <Input type="number" placeholder="e.g., 500.00" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="date"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Date</FormLabel>
              <FormControl>
                <Input type="date" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        
        {selectedCategory !== 'Salary Advance' && (
            <FormField
            control={form.control}
            name="description"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Description</FormLabel>
                <FormControl>
                    <Textarea placeholder="e.g., Monthly shop rent" {...field} />
                </FormControl>
                <FormMessage />
                </FormItem>
            )}
            />
        )}

        <DialogFooter>
          <Button type="submit">{isEditMode ? 'Save Changes' : 'Add Expense'}</Button>
        </DialogFooter>
      </form>
    </Form>
  );
}
