'use client';

import React, { useState } from 'react';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from '@/components/ui/dialog';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import type { InventoryItem } from '@/lib/data';

type AdjustStockDialogProps = {
  isOpen: boolean;
  onOpenChange: (isOpen: boolean) => void;
  item: InventoryItem;
  onSubmit: (itemId: string, newQuantity: number) => void;
};

const formSchema = z.object({
  newQuantity: z.coerce.number().min(0, 'Quantity cannot be negative.'),
});

type AdjustStockFormValues = z.infer<typeof formSchema>;

export function AdjustStockDialog({ isOpen, onOpenChange, item, onSubmit }: AdjustStockDialogProps) {
  const form = useForm<AdjustStockFormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      newQuantity: item.quantity,
    },
  });

  const handleFormSubmit = (data: AdjustStockFormValues) => {
    onSubmit(item.id, data.newQuantity);
  };

  return (
    <Dialog open={isOpen} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle>Adjust Stock for {item.name}</DialogTitle>
          <DialogDescription>
            Current quantity: {item.quantity}m. Set the new stock level below.
          </DialogDescription>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleFormSubmit)} className="space-y-4 py-4">
            <FormField
              control={form.control}
              name="newQuantity"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>New Quantity (meters)</FormLabel>
                  <FormControl>
                    <Input type="number" placeholder="Enter new quantity" {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <DialogFooter>
              <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
                Cancel
              </Button>
              <Button type="submit">Update Stock</Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}
