// Database setup and initialization utilities
import { User } from 'firebase/auth';
import { auth } from '../firebase';
import { shopService, userService } from './service';
import { checkAndRunMigration } from './migration';
import type { ShopProfileDB, UserDB } from './types';

export interface DatabaseSetupResult {
  success: boolean;
  message: string;
  shopId?: string;
  migrationResult?: any;
}

// Initialize database for a new user/shop
export async function initializeDatabase(
  user: User,
  shopData: Omit<ShopProfileDB, 'id' | 'createdAt' | 'updatedAt' | 'ownerId'>
): Promise<DatabaseSetupResult> {
  try {
    // Create shop profile
    const shopResult = await shopService.create({
      ...shopData,
      ownerId: user.uid,
    });

    if (!shopResult.success || !shopResult.data) {
      return {
        success: false,
        message: `Failed to create shop: ${shopResult.error}`,
      };
    }

    const shopId = shopResult.data.id;

    // Create user profile
    const userData: Omit<UserDB, 'id' | 'createdAt' | 'updatedAt'> = {
      email: user.email || '',
      displayName: user.displayName || '',
      photoURL: user.photoURL || '',
      role: 'owner',
      shopId: shopId,
      isActive: true,
    };

    const userResult = await userService.create(userData);

    if (!userResult.success) {
      return {
        success: false,
        message: `Failed to create user profile: ${userResult.error}`,
      };
    }

    // Check and run migration if needed
    const migrationResult = await checkAndRunMigration(shopId, user.uid);

    return {
      success: true,
      message: 'Database initialized successfully',
      shopId: shopId,
      migrationResult: migrationResult,
    };
  } catch (error: any) {
    console.error('Database initialization failed:', error);
    return {
      success: false,
      message: `Database initialization failed: ${error.message}`,
    };
  }
}

// Get or create user shop context
export async function getUserShopContext(user: User): Promise<{
  shopId: string | null;
  userRole: string | null;
  isActive: boolean;
}> {
  try {
    const userResult = await userService.getById(user.uid);
    
    if (userResult.success && userResult.data) {
      return {
        shopId: userResult.data.shopId,
        userRole: userResult.data.role,
        isActive: userResult.data.isActive,
      };
    }
    
    return {
      shopId: null,
      userRole: null,
      isActive: false,
    };
  } catch (error) {
    console.error('Failed to get user shop context:', error);
    return {
      shopId: null,
      userRole: null,
      isActive: false,
    };
  }
}

// Check if database setup is required
export async function isDatabaseSetupRequired(): Promise<boolean> {
  const user = auth.currentUser;
  if (!user) return true;
  
  const context = await getUserShopContext(user);
  return !context.shopId;
}

// Validate database connection
export async function validateDatabaseConnection(): Promise<boolean> {
  try {
    // Try to read from a collection to test connection
    const result = await shopService.getMany([], { limit: 1 });
    return result.success;
  } catch (error) {
    console.error('Database connection validation failed:', error);
    return false;
  }
}
